/**
 * Chrome MV3 Content Script Shim
 */
// Inject marker element IMMEDIATELY to prove content script runs
(function() {
    var marker = document.createElement('div');
    marker.id = 'carbonguru-shim-loaded';
    marker.style.display = 'none';
    marker.setAttribute('data-loaded', Date.now());
    (document.body || document.documentElement).appendChild(marker);
})();
console.log('[CarbonGuru Shim] Script starting...');

(function() {
    'use strict';

    console.log('[CarbonGuru Shim] Inside IIFE, checking browser object...');
    console.log('[CarbonGuru Shim] typeof browser:', typeof browser);
    console.log('[CarbonGuru Shim] browser.runtime exists:', typeof browser !== 'undefined' && !!browser.runtime);

    if (typeof browser === 'undefined') {
        console.error('[CarbonGuru Shim] ERROR: browser is undefined - polyfill not loaded?');
        return;
    }

    const originalSendMessage = browser.runtime.sendMessage.bind(browser.runtime);
    console.log('[CarbonGuru Shim] originalSendMessage bound successfully');
    const MAX_RETRIES = 3;
    const RETRY_DELAY_MS = 500;
    const TIMEOUT_MS = 5000;

    /**
     * Send message with retry logic for MV3 service worker lifecycle
     */
    async function sendMessageWithRetry(message, retries = 0) {
        return new Promise((resolve, reject) => {
            const timeoutId = setTimeout(() => {
                reject(new Error('Message timeout - service worker may not be ready'));
            }, TIMEOUT_MS);

            originalSendMessage(message).then(
                (response) => {
                    clearTimeout(timeoutId);
                    resolve(response);
                },
                (error) => {
                    clearTimeout(timeoutId);
                    // Check if error is due to disconnected port or invalidated context (service worker not ready)
                    if (retries < MAX_RETRIES &&
                        (error.message?.includes('disconnected') ||
                         error.message?.includes('Receiving end does not exist') ||
                         error.message?.includes('Extension context invalidated'))) {
                        console.log(`[CarbonGuru Shim] Retrying message (${retries + 1}/${MAX_RETRIES})...`);
                        setTimeout(() => {
                            sendMessageWithRetry(message, retries + 1).then(resolve, reject);
                        }, RETRY_DELAY_MS);
                    } else {
                        reject(error);
                    }
                }
            );
        });
    }

    browser.runtime.sendMessage = function(message, callbackOrOptions, callback) {
        // Detect callback-style: sendMessage(message, callback)
        if (typeof callbackOrOptions === 'function') {
            callback = callbackOrOptions;
            callbackOrOptions = undefined;
        }

        // If callback provided, convert to Promise-style with retry
        if (typeof callback === 'function') {
            sendMessageWithRetry(message).then(
                (response) => callback(response),
                (error) => {
                    console.warn('[CarbonGuru Shim] sendMessage error:', error.message);
                    callback(undefined);
                }
            );
            return true; // Indicate async response (MV2 pattern)
        }

        // Otherwise, use Promise-style with retry
        return sendMessageWithRetry(message);
    };

    console.log('[CarbonGuru Shim] Content script callback shim loaded (with retry support)');

    /**
     * Keepalive utility for MV3 service worker
     *
     * During active operations (like analysis polling), content scripts can
     * call cgKeepalive.start() to keep the service worker alive, and
     * cgKeepalive.stop() when done.
     *
     * Usage in content.js:
     *   if (window.cgKeepalive) window.cgKeepalive.start();
     *   // ... do polling ...
     *   if (window.cgKeepalive) window.cgKeepalive.stop();
     */
    const cgKeepalive = {
        _port: null,
        _pingInterval: null,

        start() {
            if (this._port) return; // Already connected

            try {
                this._port = browser.runtime.connect({ name: 'cg_keepalive' });
                this._port.onDisconnect.addListener(() => {
                    this._port = null;
                    this._clearPing();
                });

                // Send periodic pings to keep connection alive
                this._pingInterval = setInterval(() => {
                    if (this._port) {
                        try {
                            this._port.postMessage({ type: 'ping' });
                        } catch (e) {
                            this._clearPing();
                        }
                    }
                }, 20000); // Ping every 20s

                console.log('[CarbonGuru Shim] Keepalive started');
            } catch (e) {
                console.warn('[CarbonGuru Shim] Keepalive connect failed:', e);
            }
        },

        stop() {
            this._clearPing();
            if (this._port) {
                try {
                    this._port.disconnect();
                } catch (e) {}
                this._port = null;
            }
            console.log('[CarbonGuru Shim] Keepalive stopped');
        },

        _clearPing() {
            if (this._pingInterval) {
                clearInterval(this._pingInterval);
                this._pingInterval = null;
            }
        }
    };

    // Expose to window for content.js to use
    window.cgKeepalive = cgKeepalive;
})();
